package gov.va.med.mhv.common.jwt;

import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.RSAPrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Config object to manage the preferred JWT signing settings.
 *
 * Created by Michael Ramirez (michael@apothesource.com) on 4/20/17.
 */
public class JwtSigner {
	private static final Logger log = LoggerFactory.getLogger(JwtSigner.class);
	private static final String PKCS8_RSA_PRIVATE_KEY_PREFIX = "-----BEGIN PRIVATE KEY-----";
	private static final String PKCS8_RSA_PRIVATE_KEY_SUFFIX = "-----END PRIVATE KEY-----";

	public static RSAPrivateKey loadPrivateKey(String privateKeyPEM) {
		privateKeyPEM = privateKeyPEM.replace(PKCS8_RSA_PRIVATE_KEY_PREFIX, "")
				.replace(PKCS8_RSA_PRIVATE_KEY_SUFFIX, "").replaceAll("\\s", "");

		byte[] privateKeyDER;
		try {
			privateKeyDER = Base64.decodeBase64(privateKeyPEM);
		} catch (IllegalArgumentException e) {
			throw new RuntimeException("Invalid Base64 in JWT private key.", e);
		}

		try {
			KeyFactory keyFactory = KeyFactory.getInstance("RSA");
			return (RSAPrivateKey) keyFactory.generatePrivate(new PKCS8EncodedKeySpec(privateKeyDER));
		} catch (NoSuchAlgorithmException e) {
			log.error("FATAL: Error loading JWT signing private key- NoSuchAlgorithmException while trying to get an "
					+ "instance of the RSA key factory. This is likely due to a non-standard or severely mis-configured "
					+ "Java VM.", e);
			throw new RuntimeException("FATAL: Failed to load the JWT signing private key.", e);
		} catch (InvalidKeySpecException e) {
			log.error("FATAL: Error loading JWT signing private key- InvalidKeySpecException trying to get an instance "
					+ "of the RSA key factory. Please check that the key is PKCS8 formatted. It should start with "
					+ "'-----BEGIN PRIVATE KEY-----' and end with '-----END PRIVATE KEY----'.", e);
			throw new RuntimeException("FATAL: Failed to load the JWT signing private key.", e);
		}

	}

}
